/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xidentos.c
 *       Identify Operating System at compile/run time
 * -----------------------------------------------------------------*/

/*
Abstract:
    This file includes all functions necessary to determine the operating
    system "intended" at compile time and actually in use at run time.
    i.e. "Compiled For" is simply a question of defines but 
    "Run Time" returns the actual OS in use.

Notes:
    ** IMPORTANT **
    The "true" operating system is returned except if
    compiled for DOS (will always return DOS even if running in
    a window under NT etc.)

Environments:
    Unix, AIX, DOS, Win 3.x, Win 95/98, Win NT

***/

/* 
// Make the compiler defines a little less messy 
// AND (MOST IMPORTANTLY) take care of multiple defines.
*/

#if defined(UNIX)
#   define OS_IS_UNIX
#   include <stdio.h>

#elif defined(_AIX)
#   define OS_IS_AIX
#   include <stdio.h>

#elif defined(DOSW32)
#   define OS_IS_DOSW32

#elif defined(WIN32)
#   define OS_IS_WIN32
#   include <windows.h>

#   if defined(WIN95) && defined(WINNT)
#     error You CANNOT define both WIN95 AND WINNT
#   endif

#   if defined(WIN95) && defined(WIN2K)
#     error You CANNOT define both WIN95 AND WIN2K
#   endif

#   if defined(WIN2K) && !defined(WINNT)
#     error You MUST define WINNT when WIN2K is defined
#   endif

#   if defined(WIN95)
#     pragma message("Warning: Compiling with only WIN95(98) defined")
#     define OS_IS_WIN95
#   elif defined(WIN2K) && defined(WINNT)
    /* We always go here */
#ifdef BEST_DEBUG
#     pragma message("Compiling with WINNT and WIN2K defined")
#endif
#     define OS_IS_WIN2K
#     define OS_IS_WINNT
#   elif defined(WINNT)
#     pragma message("Warning: Compiling with only WINNT defined")
#     define OS_IS_WINNT
#   else
#     error Unknown Win32 Operating System
#   endif

#elif defined(WIN64)
#   define OS_IS_WIN32
#   define OS_IS_WIN2K
#   define OS_IS_WIN64
#   include <windows.h>

#elif defined(WINDOWS)
#   define OS_IS_WINDOWS
#   include <xwindows.h>

#else    /* whoops */
#   error Operating System not defined in IdentOs.c
#endif

#include <xidentos.h>

/*
// The "short" version of the function
*/
BX_OS_VERSION EXPORT GetOsVersionShort()
{
  return GetOsVersion( NULL, NULL );
}

/*
// The one and only real function in this file
// **IMPORTANT** ... 
// pszVersion MUST point to "char szVersion[80];"
//
*/
BX_OS_VERSION EXPORT GetOsVersion( char pszVersion[80], BX_OS_VERSION * pCompiledFor )

/************************* UNIX ***************************/
#ifdef  OS_IS_UNIX
{
  if( pCompiledFor )
  {
      *pCompiledFor = OS_UNIX;
  }
  if( pszVersion )
  {
      BESTX_SPRINTF(pszVersion, "Unix");
  }
  return OS_UNIX;
}
#endif

#ifdef  OS_IS_AIX
{
  if( pCompiledFor )
  {
      *pCompiledFor = OS_AIX;
  }
  if( pszVersion )
  {
      BESTX_SPRINTF(pszVersion, "AIX");
  }
  return OS_AIX;
}
#endif

/************************* WIN3X **************************/
#ifdef  OS_IS_WINDOWS
{
  DWORD dwVersion = GetVersion();

  if( pCompiledFor )
  {
    *pCompiledFor = OS_WIN3X;
  }

  if (dwVersion < 0x80000000) 
  {
    /* Windows NT */
    if( pszVersion )
    {
      BESTX_WSPRINTF (pszVersion, "Windows NT %u.%u (Build: %u)",
                    (DWORD)(LOBYTE(LOWORD(dwVersion))),
                    (DWORD)(HIBYTE(LOWORD(dwVersion))),
                    (DWORD)(HIWORD(dwVersion)));
    }
    return OS_WINNT;
  }
  else 
  {
    if (LOBYTE(LOWORD(dwVersion))<4) 
    {
        /* Win32s */
      if( pszVersion )
      {
        BESTX_WSPRINTF (pszVersion, "Win32s %u.%u (Build: %u)",
                    (DWORD)(LOBYTE(LOWORD(dwVersion))),
                    (DWORD)(HIBYTE(LOWORD(dwVersion))),
                    (DWORD)(HIWORD(dwVersion) & ~0x8000));
      }
      return OS_WIN3X;
    } 
    else 
    {
      /* Windows 95 or 98 */
      if( pszVersion )
      {
        BESTX_WSPRINTF (pszVersion, "Windows 95/98 %u.%u (Build: %u)",
                    (DWORD)(LOBYTE(LOWORD(dwVersion))),
                    (DWORD)(HIBYTE(LOWORD(dwVersion))),
                    (DWORD)(HIWORD(dwVersion) & ~0x8000));
      }
      return OS_WIN95;
    }
  }
}
#endif


/************************* WIN32 **************************/
#ifdef  OS_IS_WIN32
{
  OSVERSIONINFO osvi;

  if( pCompiledFor )
#   ifdef OS_IS_WIN95
      *pCompiledFor = OS_WIN95;
#   elif defined (OS_IS_WIN2K)
      *pCompiledFor = OS_WIN2K;
#   else 
      *pCompiledFor = OS_WINNT;
#   endif

  BESTX_MEMSET(&osvi, 0, sizeof(OSVERSIONINFO));
  osvi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
  GetVersionEx (&osvi);

  if (osvi.dwPlatformId == VER_PLATFORM_WIN32s)
  {
    if( pszVersion )
    {
      BESTX_WSPRINTF (pszVersion, "Win32s %d.%d (Build %d)",
                  osvi.dwMajorVersion,
                  osvi.dwMinorVersion,
                  osvi.dwBuildNumber & 0xFFFF);
    }
    return OS_WIN3X;
  }
  else
  {
    if (osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS)
    {
      if( pszVersion )
      {
        BESTX_WSPRINTF (pszVersion, "Windows 95/98 %d.%d (Build %d)",
                    osvi.dwMajorVersion,
                    osvi.dwMinorVersion,
                    osvi.dwBuildNumber & 0xFFFF);
      }
      return OS_WIN95;
    }
    else 
    {
      if (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT)
      {
        if (osvi.dwMajorVersion<=4)
        {
          if( pszVersion )
          {
            BESTX_WSPRINTF (pszVersion, "Windows NT %d.%d (Build %d)",
                      osvi.dwMajorVersion,
                      osvi.dwMinorVersion,
                      osvi.dwBuildNumber & 0xFFFF);
          }
          return OS_WINNT;
        }
        else if (osvi.dwMajorVersion==5)
        {
          if( pszVersion )
          {
            BESTX_WSPRINTF (pszVersion, "Windows 2K %d.%d (Build %d)",
                      osvi.dwMajorVersion,
                      osvi.dwMinorVersion,
                      osvi.dwBuildNumber & 0xFFFF);
          }
          return OS_WIN2K;
        }
        else if (osvi.dwMajorVersion==6)
        {
          if( pszVersion )
          {
            BESTX_WSPRINTF (pszVersion, "Windows Vista %d.%d (Build %d)",
                      osvi.dwMajorVersion,
                      osvi.dwMinorVersion,
                      osvi.dwBuildNumber & 0xFFFF);
          }
          return OS_WINVISTA;
        }
        else
        {
          return OS_UNKNOWN;
        }
      }
      else
      {
        return OS_UNKNOWN;
      }
    }
  }
}
#endif

/******* DOSW32 DOS Watcom 32 bit ****/

#ifdef  OS_IS_DOSW32
{
  if( pCompiledFor )
  {
      *pCompiledFor = OS_DOSW32;
  }
  if( pszVersion )
  {
      BESTX_SPRINTF(pszVersion, "Dos32bit");
  }
  return OS_DOSW32;
}
#endif
